//
//  OobeeLibrary.h
//  OobeeLibrary
//
//  Created by geotech on 7/23/14.
//  Copyright (c) 2014 Geocomply. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>
#import "AdvancedMerchants.h"
#import "OobeeNetworkKit.h"

typedef void(^OobeeAPNHandler)(NSError* error, BOOL success);

@protocol OobeeLibraryPostNotification <NSObject>
    @optional
    /**
     *    Implement this to be notified via post notification by Oobee SDK when
     *    server decides to unpair a specified merchant
     *
     *    @param notification notification.userInfo contains the information about merchant which was unpaired from server
     *           The information of merchant should be gotten as below
     *           Merchant * unpairedMerchant = [notification.userInfo objectForKey:@"merchant"]
     */
-(void) didUnregisterByServerSide: (NSNotification *) notification;

    /**
     *    Implement this to be notified via post notification when
     *
     *    @param notification notification.userInfo contains the error code if any. and error code is null
     *           for success. The error code should be gotten as below
     *           NSError * error = [notification.userInfo objectForKey:@"error"]
     */
-(void) didGetGeolocationResult: (NSNotification *) notification;

    /**
     *    Implement this to be notified when finish the geolocation submission
     *
     *    @param notification notification.userInfo contains the error code if any. and error code is null
     *           for success. The error code should be gotten as below
     *           NSError * error = [notification.userInfo objectForKey:@"error"]
     */
-(void) didGetGeolocationSubmissionResult: (NSNotification *) notification;

    /**
     *    Implement this to be notified when start getting geolocation
     *
     *    @param notification no any param need for this method
     */
-(void) didStartGeolocation : (NSNotification *) notification;

    /**
     *    Implement this to be notified when location service is off. The SDK
     *    wait for location service on in 4 minutes
     */
-(void) didGetLocationServiceOff;

    /**
     *    Inform the status of result of registration via web service
     *
     *    @param notification contains the error code if any
     */

-(void) didGetRegisterResultViaWebService : (NSNotification *) notification;
    /**
     *    Inform the status of result of proximity checking via web service
     *
     *    @param notification contains the error code if any and the merchant information
     *    error, @"error", merchant, @"merchant
     */
-(void) didGetProximityResultViaWebService : (NSNotification *) notification;
    @end

@protocol OobeeLibraryDelegate<NSObject>
    @optional

    /**
     *  Notify the status of APN registration
     *
     *  @param errorCode the information of error. nil for success
     */
- (void) didGetRegisterDeviceWithOobeeServerResult:(NSError*) errorCode;

    /**
     *  Notify the status of registration of device to the guvnr server
     *
     *  @param errorCode the information of returned error. nil for success
     *  @param merchant  the detail infomation about registered client
     */
- (void) didGetRegisterDeviceWithMerchantResult:(NSError*) errorCode
                                       Merchant:(Merchant*) merchant;

    /**
     *  Notify about the proximity request just received via APN
     *
     *  @param merchant the detail infomation about registered client
     */
- (void) didSaveProximityRequest:(Merchant*) merchant;

    /**
     *  Notify about the proximity request just received via APN
     *
     *  @param merchant the detail infomation about registered client
     */
- (void) didFailToSaveProximityRequest:(NSError*) merchant;

    /**
     *  Notify the status of proximity checking
     *
     *  @param errorCode the information of returned error. nil for success
     *  @param merchant  the detail infomation about registered client
     */
- (void) didGetProximityCheckingResult:(NSError*) errorCode
                              Merchant:(Merchant*) merchant;

    /**
     *  Notify the unregistration status
     *
     *  @param errorCode the information of returned error. nil for success
     *  @param merchant  the detail infomation about registered client
     */
-(void) didGetUnregisterDeviceWithMerchantResult:(NSError*) errorCode
                                        Merchant:(Merchant*) merchant;


    /**
     *  Notify and send the detail information about failure on network
     *
     *  @param errorCode the information of returned error. nil for success
     */
-(void) didGetFailureOnNetwork:(NSError*) errorCode;

-(BOOL) didReceiveDataOnUltrasoundAndShouldContinueRequestToServer;// return NO to stop request.
-(BOOL) didReceiveDataOnQRCodeAndShouldContinueRequestToServer;// return NO to stop request.
-(void) QRScanDidCancel;

    @end


@interface OobeeLibrary : NSObject

    @property (nonatomic, assign) id<OobeeLibraryDelegate> delegate;
+ (instancetype)shared;
- (void)setDelegate:(id<OobeeLibraryDelegate>)delegate;
- (void)runOnPort:(NSInteger)port onComplete:(void (^)(NSError* error))completion;
- (void)stopServer;

#pragma mark sdk public method

- (void)processOobeeAPN:(NSDictionary*)payload completionHandler:(OobeeAPNHandler)completion;
- (void)processToken:(NSString*)token completionHandler:(void (^)(NSError* error))handler;

- (void)registerDevice:(NSString*)token to:(NSURL*)server handler:(OobeeUtilityHandler)handler;
- (void)pair:(NSString*)pairCode to:(NSURL*)server handler:(OobeeUtilityHandler)handler;
- (void)sync:(NSString*)token to:(NSURL*)server handler:(OobeeUtilityHandler)handler;
- (void)proximity:(NSString*)pairCode to:(NSURL*)server handler:(OobeeUtilityHandler)handler;
- (void)geolocation:(NSDictionary*)payload to:(NSURL*)server handler:(OobeeUtilityHandler)hanler;
- (void)unpair:(NSString*)uuid to:(NSURL*)server handler:(OobeeUtilityHandler)handler;
- (void)updateMerchatInfo:(NSDictionary*)info handler:(OobeeUtilityHandler)handler;
    /**
     *  Call to register the device with APN service
     *
     *  @param deviceTokenId the device token ID used for push notification
     *  @param requester     the address of object which calls this method. The result of action will be return to this address
     *  @return Error if any
     */
-(NSError*) registerDeviceWithOobeeServer:(NSString*) deviceTokenId Requester:(id<OobeeLibraryDelegate>) requester;

    /**
     *  Call to register device with guvnr server
     *
     *  @param pairCode  the pair code used to register device
     *  @param requester the address of object which calls this method
     *  @return Error if any
     */
-(NSError*) registerDeviceWithMerchant:(NSString*) pairCode  Requester:(id<OobeeLibraryDelegate>) requester;

    /**
     *  Call to submit the paircode and related information for proximity checking
     *
     *  @param pairCode  the paircode for proximity checking
     *  @param deviceUID deviceUID for client which request for proximity checking
     *  @param requester the address of object which calls this method
     *  @return Error if any
     */
-(NSError*) submitProximityCheckingCode: (NSString*) pairCode
                              deviceUID:(NSString*) deviceUID
                              Requester:(id<OobeeLibraryDelegate>) requester;

    /**
     *  Call  to unregister device with specified client (determine by deviceUID)
     *
     *  @param deviceUID deviceUID for client which will be unregistered
     *  @param requester the address of object which calls this method
     *  @return Error if any
     */
-(NSError*) unRegisterDeviceWithMerchant: (NSString*) deviceUID
                               Requester:(id<OobeeLibraryDelegate>) requester;

    /**
     *  Call to process the push notification information
     *
     *  @param payload the payload of push notification
     *
     *  @return Error if any
     */
-(NSError *) processRemoteNotification:(NSDictionary*) payload
                             Requester:(id<OobeeLibraryDelegate>) requester;

    /**
     *    Process package from socket client
     *
     *    @param url   called URL
     *    @param error error returned if any
     *
     *    @return return the content of replication to client
     */
-(NSString*) processSocketPackage:(NSURL*) url
                        Requester:(id<OobeeLibraryDelegate>) requester
                            Error:(NSError **) error;

    /**
     *    Process custom url link
     *
     *    @param url   called URL
     *    @param error error returned if any
     *
     *    @return return the content of replication to client
     */
-(NSString*) processCustomUrl:(NSURL*) url
                    Requester:(id<OobeeLibraryDelegate>) requester
                        Error:(NSError **) error;


#pragma mark Merchant method
    /**
     *  Call to get the merchant information by the merchant ID (client ID)
     *
     *  @param merchantID the ID of merchant need to request
     *
     *  @return full of merchant information
     */
-(Merchant *)getMerchantWithID:(NSString *)merchantID Error:(NSError **) error;

    /**
     *  Get all the merchant saved in SDK database
     *
     *  @return array of merchant objects
     */
-(NSArray *)getAllMerchants: (NSError **) error;

    /**
     *  Get all the merchants which have unprocessed requests for proximity checking
     *
     *  @return return the list of merchant which have unprocessed requests for proximity checking
     */
-(NSArray*) getPendingProximityMerchants:(NSError **) error;

    /**
     *  Get the merchant specified by deviceUID
     *
     *  @param deviceUID the deviceUID to specify merchant
     *
     *  @return merchant which has above device UID
     */
-(Merchant*)getMerchantWithDeviceUID:(NSString*)deviceUID Error:(NSError **) error;

-(void)startUltrasoundRegisterDeviceWithRequester:(id<OobeeLibraryDelegate>)requester;
-(void)startUltrasoundSubmitProximityCheckingDeviceUID:(NSString*) deviceUID
                                             Requester:(id<OobeeLibraryDelegate>) requester;
-(void)stopListen;


-(void)startQRCodeRegisterDeviceWithRequester:(id<OobeeLibraryDelegate>)requester presentOnViewController:(UIViewController*)vc;
-(void)startQRCodeSubmitProximityCheckingDeviceUID:(NSString*) deviceUID
                                         Requester:(id<OobeeLibraryDelegate>) requester presentOnViewController:(UIViewController*)vc;

    @end
