//
//  GeoComplySDK.h
//  GeoComplySDK
//
//  Created by Hao Nguyen on 3/29/19.
//

#import <UIKit/UIKit.h>
#import "GeoComplyDef.h"

//! Project version number for GeoComplySDK.
FOUNDATION_EXPORT double GeoComplySDKVersionNumber;

//! Project version string for GeoComplySDK.
FOUNDATION_EXPORT const unsigned char GeoComplySDKVersionString[];

/**
 These constant indicate the type of error that resulted after a geolocation request.
 */
typedef NS_ENUM(NSInteger, GCClientError) {
    GCCErrorNone                      = 0,
    
    /**
     Indicates that an unknown error occurs.
     */
    GCCErrorUnknown                   = 600,
    
    /**
     Indicates a network connection error.
     */
    GCCErrorNetworkConnection         = 602,
    
    /**
     Indicates an error when communicating with GeoComply Server.
     */
    GCCErrorServerCommunication       = 603,
    
    /**
     Indicates that this client has been suspended.
     */
    GCCErrorClientSuspended           = 604,
    
    /**
     Indicates that GeoComply iOS SDK solution has been disabled.
     */
    GCCErrorDisabledSolution          = 605,
    
    /**
     Indicates that license string has invalid format.
     */
    GCCErrorInvalidLicenseFormat      = 606,
    
    /**
     Indicates that Client's license is unauthorized.
     */
    GCCErrorClientLicenseUnauthorized = 607,
    
    /**
     Indicates that Client's license is expired.
     */
    GCCErrorLicenseExpired            = 608,
    
    /**
     Indicates that Custom Field(s) is(are) invalid.
     */
    GCCErrorInvalidCustomFields       = 609,
    
    /**
     Indicates that Location Service is turn off and geolocation request is cancelled.
     */
    GCCErrorRequestCanceled           = 611,
    
    /**
     Indicates that device's memory is low.
     */
    GCCErrorLowMemory                 = 612,
    
    /**
     Indicates that a geolocation request is in progress.
     */
    GCCErrorGeolocationInProgress     = 614,
    
    /**
     Indicates that GeoComply iOS SDK is updating indoor location.
     */
    GCCErrorIsUpdatingLocation        = 630,
    
    /**
     Indicates that application did enter background.
     */
    GCCErrorBackgroundMode            = 634,
    
    /**
     Indicates that input for User ID / Phone Number / Geolocation Reason / Custom Fields is (are) in invalid format.
     */
    GCCErrorInvalidInput              = 635,
    
    /**
     Indicates that the geolocation request has been modified during submission by another party.
     */
    GCCErrorInvalidHMAC               = 640
};

/**
 The GCError object manages information about the details of a failed geolocation.
 
 GCError will be return via -didGeolocationFailed: delegate method.
 
 @@see -didGeolocationFailed:
 */
@interface GCError: NSObject

/**
 Returns error code when a geolocation request failed.
 
 @@see GCClientError
 */
@property (nonatomic, readonly) GCClientError code;

/**
 Returns description when a geolocation request failed.
 */
@property (nonatomic, readonly, nullable) NSString* description;

/**
 A determination flag for app to retry calling a geolocation request.
 */
@property (nonatomic, readonly) BOOL shouldRetry;

@end


/**
 The GCClientDelegate object manages protocol-allow for an object to receive callbacks
 regarding the success or failure of a geolocation request.
 
 When a geolocation successes, -didGeolocationAvailable: is called.
 Otherwise, -didGeolocationFailed: is called instead.
 
 @warning Delegate property of GeoComplyClient instance can not be nil. Otherwise, callbacks
 will be discarded.
 */
@protocol GCClientDelegate<NSObject>

/**
 Tells the delegate a geolocation request is success.
 
 @param data Encrypted geolocation data returned from GeoComply Server. For decryption, use
 provided AES KEY & IV.
 */
- (void)didGeolocationAvailable:(nonnull NSString*)data;

/**
 Tells the delegate a geolocation request is failed.
 
 @param error A GCError object.
 
 @@see GCError
 */
- (void)didGeolocationFailed:(nonnull GCError*)error;

@optional

/**
 Tells the delegate that there's a log message from GeoComply iOS SDK.
 
 @param message A log message from GeoComply iOS SDK.
 */
- (void)updateLog:(nonnull NSString*)message;

/**
 Tells the delegate that Location Service is disabled and whether GeoComply
 iOS SDK should cancel current geolocation request.
 
 @return A determination flag whether GeoComply iOS SDK should cancel current
 geolocation request. Default is NO.
 
 @warning If this protocol is not implemented. GeoComply iOS SDK will still send
 current geolocation request to GeoComply Server for location determination. This
 is default behavior for GeoComply iOS SDK.
 */
- (BOOL)didLocationServiceDisabled;

/**
 Tells the delegate that GeoComply iOS SDK has stopped updating indoor location
 completely.
 
 @param info The detailed information.
 */
- (void)didStopUpdating:(nullable NSDictionary*)info;

/**
 Telss the delegate that GeoCompoly iOS SDK is asking to display a message to turn
 on Bluetooth.
 
 @param message The message for app to display.
 */
- (void)didTurnOffBluetooth:(nonnull NSString*)message;

@end

/**
 The GCCustomFields object contains extra information about a geolocation request.
 
 @warning If no GCCustomFields instance is not set to GeoComplyClient object. There's
 not any custom field is included in geolocation request.
 */
@interface GCCustomFields : NSObject
/**
 Sets a new value for a key in GCCustomFields instance. If a key doesn't exist, it will
 be created.
 
 @param fieldValue New value. Unless value is nil, in which case send -removeValueForKey:
 @param fieldKey Key needs to set value. This must not be nil.
 */
- (nullable NSError*)setValue:(nonnull NSString*)fieldValue forKey:(nonnull NSString*)fieldKey;

/**
 Returns value for a key.
 
 @return Value for input key.s
 
 @param fieldKey Field that needs to retrieve value.
 */
- (nullable NSString*)valueForKey:(nonnull NSString*)fieldKey;

/**
 Returns all keys in GCCustomFields instance.
 
 @return An array of keys.
 */
- (nullable NSArray*)allKeys;

/**
 Remove a key and its value from GCCustomFields instance.
 
 @param fieldKey Field that needs to be removed.
 */
- (void)removeValueForKey:(nonnull NSString*)fieldKey;

/**
 Remove all key and values from GCCustomFields instance.
 */
- (void)removeAllValues;

@end

/**
 The GeoComplyClient object manages all geolocation request call
 */
@interface GeoComplyClient : NSObject

/**
 Sets or return current user id for geolocation request.
 */
@property (nonatomic, copy, nonnull) NSString* userId;

/**
 Sets or return geolocation reason for geolocation request.
 */
@property (nonatomic, copy, nonnull) NSString* geolocationReason;

/**
 Sets or return user phone number for geolocation request.
 */
@property (nonatomic, copy, nullable) NSString* userPhoneNumber;

/**
 Sets or return GCCustomFields for geolocation request.
 
 @@see GCCustomFields.
 */
@property (nonatomic, readonly, nullable) GCCustomFields* customFields;

/**
 Sets or returns protocol for GCClientDelegate.
 
 @@see GCClientDelegate.
 */
@property (nonatomic, weak, nullable) id<GCClientDelegate> delegate;

/**
 Sets license string to GeoComplyClient instance.
 
 @param license The license string retrived from GeoComply Server.
 @param errorPtr An error pointer for error diagnostic.
 
 @return A boolean to indicate whether the license has been set successfully.
 */
- (BOOL)setLicense:(nonnull NSString*)license error:(NSError* _Nullable __autoreleasing* _Nullable)errorPtr;

/**
 Calls a geolocation request.
 
 @param error An error pointer for error diagnostic.
 
 @return A boolean to indicate whether the geolocation request is called successfully.
 */
- (BOOL)requestGeolocation:(NSError* _Nullable __autoreleasing* _Nullable)error;

/**
 Calls a geolocation request with timeout.
 
 @param error An error pointer for error diagnostic.
 @param timeout A timeout value for geolocation request.
 
 @return A boolean to indicate whether the geolocation request is called successfully.
 
 @warning If timeout value is equal to 0, GeoComply iOS SDK will use timeout value configured on GeoComply Server.
 This is equivalent to -requestGeolocation:
 */
- (BOOL)requestGeolocation:(NSError* _Nullable __autoreleasing* _Nullable)error timeout:(NSInteger)seconds;

/**
 Creates and returns a GeoComplyClient object.
 */
+ (nonnull GeoComplyClient*) instance;

/**
 Handles memory warning notification.
 
 @warning This will cancel current geolocation request and returns error GCCErrorLowMemory via GCClientDelegate protocol -didGeolocationFailed:.
 
 @see -didGeolocationFailed:
 */
+ (void)handleMemoryWarningForClient:(nonnull GeoComplyClient*)client;

/**
 Calls indoor geolocation updates.
 
 @warning Calls this method will auto reschedule next indoor geolocation updates. No further scheduling is required.
 */
- (BOOL)startUpdating:(NSError* _Nullable __autoreleasing* _Nullable)error;

/**
 Stops indoor geolocation updates.
 
 @warning Calls this method will invoke GCClientDelegate protocol -didStopUpdating:.
 
 @see -didStopUpdating:.
 */
- (void)stopUpdating;

/**
 Returns a boolean to indicate whether GeoComplyClient iOS SDK is updating indoor geolocation.
 */
- (BOOL)isUpdating;

/**
 Returns UUID of current geolocation request.
 
 @warning This method only returns valid UUID string when triggering -requestGeolocation:.
 
 @see -requestGeolocation:.
 */
- (nullable NSString*)currentRequestUUID;

@end
